<?php
header('Content-Type: application/json');

// ======== 🔐 CONFIGURATION ========
define('AUTH_TOKEN', 'your_secure_token_here'); // Secure endpoint token
define('ALERT_THRESHOLD', 5000); // Profit alert threshold (₦)
define('PROFIT_LOG_FILE', __DIR__ . '/../profits.json');
define('ALERT_LOG_DIR', __DIR__ . '/../logs');

// ======== 🔔 TELEGRAM CONFIG ========
define('TELEGRAM_BOT_TOKEN', 'YOUR_TELEGRAM_BOT_TOKEN');
define('TELEGRAM_CHAT_ID', 'YOUR_TELEGRAM_CHAT_ID');

// ======== 🕒 TIME ========
$timestamp = date('Y-m-d H:i:s');

// ======== 📊 Simulated Exchange Rates (Replace with live API in production) ========
$exchanges = [
    "KuCoin" => [ "buy" => rand(1290, 1320), "sell" => rand(1400, 1450) ],
    "Paxful" => [ "buy" => rand(1285, 1310), "sell" => rand(1380, 1430) ],
    "Bybit"  => [ "buy" => rand(1275, 1305), "sell" => rand(1390, 1445) ],
    "Gate.io"=> [ "buy" => rand(1300, 1330), "sell" => rand(1410, 1460) ]
];

// ======== 💰 Profit & ROI Calculation ========
$buyAmount = 1000;
$sources = [];
$totalProfit = 0;
$topExchange = null;
$highestProfit = PHP_INT_MIN;

foreach ($exchanges as $name => $rate) {
    $profit = $rate['sell'] - $rate['buy'];
    $roi = $profit > 0 ? ($profit / $rate['buy']) * 100 : 0;

    $sources[] = [
        "source" => $name,
        "buy"    => $rate['buy'],
        "sell"   => $rate['sell'],
        "profit" => round($profit, 2),
        "roi"    => round($roi, 2)
    ];

    $totalProfit += $profit;

    // Track highest-profit exchange
    if ($profit > $highestProfit) {
        $highestProfit = $profit;
        $topExchange = $name;
    }

    // ======== 🚨 Profit Alert System ========
    if ($profit >= ALERT_THRESHOLD) {
        $alertText = "🚨 *Profit Alert!*\n"
            . "*Exchange:* {$name}\n"
            . "*Buy:* ₦" . number_format($rate['buy'], 2) . "\n"
            . "*Sell:* ₦" . number_format($rate['sell'], 2) . "\n"
            . "*Profit:* ₦" . number_format($profit, 2) . "\n"
            . "*ROI:* " . number_format($roi, 2) . "%\n"
            . "*Time:* {$timestamp}";

        // Telegram alert
        $telegramUrl = "https://api.telegram.org/bot" . TELEGRAM_BOT_TOKEN . "/sendMessage";
        $telegramPayload = [
            'chat_id'    => TELEGRAM_CHAT_ID,
            'text'       => $alertText,
            'parse_mode' => 'Markdown'
        ];
        @file_get_contents($telegramUrl . '?' . http_build_query($telegramPayload));

        // Log alerts to file
        if (!is_dir(ALERT_LOG_DIR)) mkdir(ALERT_LOG_DIR, 0777, true);
        $logFile = ALERT_LOG_DIR . '/' . strtolower(str_replace('.', '', $name)) . '.log';
        $logLine = "[" . $timestamp . "] " . str_replace("\n", " | ", strip_tags($alertText)) . "\n";
        file_put_contents($logFile, $logLine, FILE_APPEND | LOCK_EX);
    }
}

// ======== 🤖 AI Profit Prediction ========
// Simple moving average trend based on last 5 entries
$existing = file_exists(PROFIT_LOG_FILE) ? json_decode(file_get_contents(PROFIT_LOG_FILE), true) : [];
$recentProfits = [];

foreach (array_slice($existing, -5) as $prev) {
    $recentProfits[] = array_sum(array_column($prev['sources'], 'profit'));
}

$aiPrediction = null;
if (count($recentProfits) > 0) {
    $avgChange = 0;
    for ($i = 1; $i < count($recentProfits); $i++) {
        $avgChange += ($recentProfits[$i] - $recentProfits[$i - 1]);
    }
    $avgChange = $avgChange / max(1, (count($recentProfits) - 1));
    $aiPrediction = round(end($recentProfits) + $avgChange, 2);
}

// ======== 📈 Final Entry Structure ========
$entry = [
    "timestamp"        => $timestamp,
    "buy"              => $buyAmount,
    "sources"          => $sources,
    "totalProfit"      => round($totalProfit, 2),
    "topExchange"      => $topExchange,
    "aiPredictionNext" => $aiPrediction
];

// ======== 💾 Save to File (keep last 100 records) ========
$existing[] = $entry;
$existing = array_slice($existing, -100);
file_put_contents(PROFIT_LOG_FILE, json_encode($existing, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE), LOCK_EX);

// ======== ✅ API Response ========
echo json_encode([
    'status'        => 'ok',
    'entry'         => $entry,
    'totalProfit'   => round($totalProfit, 2),
    'accumulated'   => round(array_sum(array_map(fn($e) => $e['totalProfit'], $existing)), 2),
    'topExchange'   => $topExchange,
    'aiPrediction'  => $aiPrediction
]);
?>
